"""Utilities used by various of Doug Lee's Python programs.

Copyright (C) 2008-2024 Doug Lee

This program is free software: you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation, either version 3 of the License, or (at your
option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with this program.  If not, see <http://www.gnu.org/licenses/>.

"""

import os, sys, subprocess, re, shutil

def safeCall(*args, **qwargs):
	"""Call that retries on OSError 11, which happens on Cygwin 1.8.
	safeCall and callWithRetry are both provided here for historical reasons.
	"""
	maxtries = 20
	while maxtries:
		maxtries -= 1
		try:
			return subprocess.call(*args, **qwargs)
		except OSError as e:
			if e.errno != 11:
				raise

def callWithRetry(func, *args, **kwargs):
	"""For Cygwin 1.8 on Windows:
	Forks can ffail randomly in the presence of things like antivirus software,
	because DLLs attaching to the process can cause address mapping problems.
	This function retries such calls so they don't fail.
	safeCall and callWithRetry are both provided here for historical reasons.
	"""
	i = 1
	while i <= 50:
		try:
			return func(*args, **kwargs)
		except OSError as e:
			i += 1
			print("Retrying, attempt #" +str(i))
	print("Retry count exceeded.")

class PathCache(dict):
	"""A class for caching executable paths, and also absences of them.
	"""
	_cache = {}
	@classmethod
	def getPath(cls, name):
		"""Get a path for an executable by its name, or return the null string if it is not found. Cache the result for the next call with the same name.
		"""
		p = cls._cache.get(name)
		if p is not None: return p
		p = shutil.which(name)
		cls._cache[name] = p
		return p

def hrsize(size):
	"""Return a 10-character string representing the given size succinctly in human-readable form:
		0-1023 bytes: Size suffixed with " bytes.
		0-1023 kb, meg, gig, tb: Size with two decimal places with the indicated suffix.
	All values are aligned so that the first four characters are always thousands through ones for any unit.
	"""
	if isinstance(size, str): size = int(size)
	units = ["bytes", "kb", "mb", "gb", "tb"]
	uidx = 0
	val = size
	while val > 1024 and uidx <= len(units):
		val /= 1024.0
		uidx += 1
	if uidx == 0:
		val = f"{val:4d} {units[uidx]}"
	else:
		val = f"{val:7.2f} {units[uidx]}"
	return val

def timeToSecs(tm):
	"""Return the number of seconds represented by the given possibly incomplete or improper time, like 2: or 76:88. Returns float.
	Needed for sending start times to ffmpeg.
	"""
	parts = tm.split(":")
	# Fix null segments. This is for things like 33: and 4:22: but technically also handles things like 5::22.
	parts = [p if p else "0" for p in parts]
	secs = 0
	try:
		# The first one to fail stops this.
		secs += float(parts.pop())
		secs += float(parts.pop()) * 60.0
		secs += float(parts.pop()) * 3600.0
	# Unlikely but supported, days.
		secs += float(parts.pop()) * (3600.0 *24.0)
	except IndexError: pass
	return secs

def secsToTime(secs):
	"Convert seconds to hh:mm:ss."
	mm,ss = divmod(secs, 60)
	hh,mm = divmod(mm, 60)
	return "%02d:%02d:%02d" % (hh, mm, ss)

def copyTimes(path1, path2):
	"""Copy the access and modification times of path1 to path2.
	"""
	st = os.stat(path1)
	nstimes = (st.st_atime_ns, st.st_mtime_ns)
	os.utime(path2, ns=nstimes)

def isOggOpusFile(soundfile):
	"""Returns True if soundfile appears to be an Ogg Opus file.
	"""
	try:
		with open(soundfile, "rb") as f:
			bytes = f.read(512)
			if re.search(b"OpusHead", bytes): return True
	except Exception: return False
	return False

def soxFileTypes(path=None):
	"""Return a set of extensions (with leading dot) supported by the local SoX version.
	If path is given, it is the SoX instance to use; otherwise the environment PATH is used to find it.
	"Supported" here means able to decode.
	"""
	if not path: path = PathCache.getPath("sox")
	if not path: path = PathCache.getPath("sox.exe")
	proc = subprocess.run(path, stdin=subprocess.DEVNULL, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, text=True)
	stdout = proc.stdout
	formats = stdout.split("AUDIO FILE FORMATS:", 1)[1].split("\n", 1)[0].strip().split(None)
	formats = ["."+f for f in formats]
	return set(formats)

def ffmpegFileTypes(path=None):
	"""Return a set of extensions (with leading dot) supported by the local ffmpeg version.
	If path is given, it is the ffmpeg instance to use; otherwise the environment PATH is used to find it.
	If ffmpeg is not found or there is an error, the empty set is returned.
	"Supported" here means able to decode (demux, as ffmpeg puts it).
	"""
	if not path: path = PathCache.getPath("ffmpeg")
	if not path: path = PathCache.getPath("ffmpeg.exe")
	if not path: return set()
	proc = subprocess.run([path, "-formats"], stdin=subprocess.DEVNULL, stdout=subprocess.PIPE, stderr=subprocess.DEVNULL, text=True)
	lines = proc.stdout.splitlines()
	formats = set()
	started = False
	for line in lines:
		line = line.strip()
		# Ignore header material.
		if not line: continue
		if not started and "--" in line:
			started = True
			continue
		if not started: continue
		# An actual format line. Format: flags fmt[,fmt...] description.
		# Flags: D can decode, E can encode.
		try: flags,fmts,rest = line.split(None, 2)
		except ValueError: flags,fmts = line.split(None, 1)
		if "d" not in flags.lower(): continue
		fmts = fmts.split(",")
		[formats.add(f".{fmt}") for fmt in fmts]
	return formats

def platform():
	plat = sys.platform
	if plat.startswith("linux") and os.path.exists("/mnt/c/Windows"):
		plat = "wsl"
	return plat

def SoXDev(ln, io="o"):
	"""Return the platform-specific full-name translation of an abbreviated SoX-compatible device identifier.
	The optional second argument, "i" or "o" (default), means input versus output device.
	Currently only these platforms are supported: Windows, Cygwin and WSL (on Windows), MacOS (Darwin).
	Warning: These mappings are author-specific, though the VAC mappings may be of use to others.
	For an unrecognized device, i/o value, combination, or platform, the given value is simply returned unchanged.
	"""
	if ln == "-p": return ln
	io = io.lower()
	if io not in "io": return ln
	isIn = (io == "i")
	plat = platform()
	if plat.startswith("win") or plat.startswith("cygwin") or plat == "wsl":
		if ln == "-d":
			return "Microsoft Sound Mapper"
		if ln[0] in "lL" and ln[1:].isdigit():
			# str(int()) removes any leading zeros.
			return "Line {} (Virtual Audio Cable)".format(str(int(ln[1:])))
		if ln.lower() == "a":
			return "Microphone (Andrea PureAudio US" if isIn else "Speakers (Andrea PureAudio USB-"
		if ln.lower() == "c":
			return "Microphone (Arctis Nova 7)" if isIn else "Headphones (Arctis Nova 7)"
		if ln.lower() == "g":
			# No stereo input available on this headset.
			if isIn: return ln
			return "Headphones (Arctis Nova 7)"
		if ln.lower() == "h":
			return "Microphone (Logi H800 Headset)" if isIn else "Speakers (Logi H800 Headset)"
		if ln.lower() == "r":
			return "Microphone Array (Realtek(R) Au" if isIn else "Speakers/Headphones (Realtek(R)"
		if ln.lower() == "u":
			return "Line (USB Audio Device)" if isIn else "Speakers (USB Audio Device)"
		return ln
	elif plat.startswith("darwin"):
		if ln == "-d":
			# ToDo: This probably won't work in practice because the caller is likely to include "-t coreaudio" before it.
			return "-d"
		if ln[0] in "lL" and ln[1:].isdigit():
			# str(int()) removes any leading zeros.
			# ToDo: Replace with standard Loopback names.
			return "Line {} (Virtual Audio Cable)".format(str(int(ln[1:])))
		if ln.lower() == "h":
			return "Microphone (Logitech Wireless Headset)" if isIn else "Speakers (Logitech Wireless Headset)"
		if ln.lower() == "i":
			return "Built-in Input" if isIn else "Built-in Outpu"
		return ln
	return ln

class SoXFX:
	"""Handler of custom-defined SoX effect shortcut conversion into full SoX effect chains.
	Initialize with an effect chain that might include expandable elements.
	After this, the following properties will exist:
		given: The originally given effect chain, with any shortcuts unchanged.
		expanded: The expanded and SoX-ready effect chain.
		profEffects: Any effects required for a separate initial SoX run to create a noise profile. "" if not needed.
	All shortcuts defined here contain no spaces.
	Supported shortcut patterns, where <f> and <n> are a float and an int, respectively:
	h<f> or l<f> becomes highpass <f> or lowpass <f>.
	comp<f>s becomes compand .01,.01,.01,.01 -<f>,-20,0,-10
	comp<f> becomes compand .01,.01 -<f>,-20,0,-10
		(Those two allow for quick and fast-reacting compressions that do and don't allow channels to vary with respect to each other.)
	eqs<f1>[-<f2>][/<width>] becomes a range of equalizer effects; examples:
		eqs60-300 becomes equalizers 60/120/180/240/300 .0005h -180
		eqs60-300/.005h becomes the same with slightly wider cutout bands.
		These are very effective means to remove harmonics such as from a 60Hz hum.
	fmr and fms: FM sound simulation and FM broadcast signal simulation, respectively.
	sil<f> becomes silence 1 0 <f>% -1 0 <f>%
		These are noise gates to trim out periods below a sound level threshhold.
		For historical reasons, sil0<n> becomes silence 1 0 .0<n>% -1 0 .0<n>% (equivalent to specifying sil0.0<n>).
		For example, sil03 and sil.03 are the same. Omitting the dot is deprecated.
	nv<n>[@<f1>[/<f2>]] Cut out sound at and above <n>dB, possibly centering around <f1>Hz and possibly with a width of <f2>Hz. <f2> defaults to 400Hz if not given when <f1> is given.
		This is a way to keep only the background sound from a file based on sound level threshhold and possibly frequency range of non-background sound.
		It can also be used to trim out very loud sounds in a quiet file as if they never happened.
		"nv" originally stood for "no vocals" as the author used this shortcut to test individual vocal tracks for unwanted background before mixing.
	ov<n>[@<f1>[/<f2>]] Cut out sound at and below <n>dB, possibly centering around <f1>Hz and possibly with a width of <f2>Hz. <f2> defaults to 400Hz if not given when <f1> is given.
		This is the inverse of nv and originally meant "only vocal." It preserves the described sound and trims out the rest.
	nr<start>,<duration>[,<factor>]: Noise reduction with on-the-fly profile.
		start and duration are time specifications like 3:0 and 0:1, or 3.1 (seconds) and 0:0.5.
		Without factor or with a factor of 1[.0], noisered uses 0.0. If factor is less than 1, noisered uses that.
		If factor is above 1, noisered uses 0.0 and a vol <factor> is prepended to the noisered effect.
		Warning: This effect creates self.profEffects for an extra SoX process to pipe the profile into this one;
		hence, it will not work in a live input stream setting.
		The creation and piping of this process is left to the caller.
	dtmf<string>[/[<duration>][/[<delay>][/[<fade>][/[<end>]]]]]: Dial DTMF tones.
		<string> is any sequence of 0-9 # * ABCD (case insensitive).
		<duration> and <delay> are in milliseconds and specify tone and inter-tone gap duration, defaults 200 and 100.
		<fade> is the millisecond duration of the start and end fades on tones, default 20 ms.
		<end> is the duration of the final delay after the tone sequence, default 1 sec (1000 ms).
		Making this too short can cause a "ValueError: SoX process run failure" message to print.
		This effect makes the most sense with the source being -n.
	"""
	def __init__(self, given):
		self.given = given
		self.profEffects = ""
		parts = given.split()
		for i in range(0, len(parts)):
			p = parts[i]
			# h/l<f>
			p = re.sub(r'^h([\d.]+)$', r'highpass \1', p)
			p = re.sub(r'^l([\d.]+)$', r'lowpass \1', p)
			# comp<f>[s]
			p = re.sub(r'^comp([\d.]+)$', r'compand .01,.01 -\1,-20,0,-10', p)
			p = re.sub(r'^comp([\d.]+)s$', r'compand .01,.01,.01,.01 -\1,-20,0,-10', p)
			# eqs ranges.
			match = re.match(r'^eqs([\d.]+)-([\d.]+)/([\d.]+[a-zA-Z])$', p)
			if not match:
				match = re.match(r'^eqs([\d.]+)-([\d.]+)$', p)
			if not match:
				match = re.match(r'^eqs([\d.]+)$', p)
			if match:
				gr = match.groups()
				baseFreq = float(gr[0])
				if len(gr) > 1:
					endFreq = float(gr[1])
				else:
					endFreq = baseFreq
				if len(gr) > 2:
					width = gr[2]
				else:
					width = ".0005h"
				freqMult = 1
				freq = baseFreq
				p = ""
				while freq <= endFreq:
					if p: p += " "
					p += "equalizer %s %s -180" % (str(freq), width)
					freqMult = freqMult +1
					freq = freqMult *baseFreq
			# fmr and fms
			if p.lower() == "fmr" or p.lower() == "fms":
				p = (
					# ToDo: The treble effect here and the gain after it replace the original "filter 8000- 29 100" and is my guess.
					'gain -3 treble -12 8000 .4 gain -15 mcompand '
					+'"0.005,0.1 -47,-40,-34,-34,-17,-33" 100 '
					+'"0.003,0.05 -47,-40,-34,-34,-17,-33" 400 '
					+'"0.000625,0.0125 -47,-40,-34,-34,-15,-33" 1600 '
					+'"0.0001,0.025 -47,-40,-34,-34,-31,-31,-0,-30" 6400 '
					+'"0,0.025 -38,-31,-28,-28,-0,-25" '
					# The "sinc -n 255 -b 16 -17500 is a replacement in the SoX man page for a now lost original "filter" effect.
					+'gain 12 highpass 22 highpass 22 sinc -n 255 -b 16 -17500 '
					+'gain 9'
					# This next one makes the difference between FM radio sound and broadcast signal condition simulation according to the SoX man page.
					+(' lowpass -1 17801' if p=="fmr" else '')
				)
			# sil0<n> and sil<f>
			p = re.sub(r'^sil(0\d+)$', r'silence 1 0 .\1% -1 0 .\1%', p)
			p = re.sub(r'^sil([\d.]+)$', r'silence 1 0 \1% -1 0 \1%', p)
			# nv, no-vocal, number is dB at and above which to cut them out.
			# This effect also trims out the resulting silence and any other very quiet parts.
			# The first version allows for an arbitrary band to be removed preferentially.
			# The second version allows for a 400h-wide band to be removed preferentially.
			sil = "silence 1 0 .001% -1 0 .001%"
			p = re.sub(r'^nv(\d+)@([\d.]+)/([\d.]+)$', r'vol .01 equalizer \2 \3h 55 compand .0001,.05 6:-\1.0001,-\1,-inf,0,-inf 0 0 .0015 '+sil+r' -1 .022 .001% equalizer \2 \3h -55 vol 100', p)
			p = re.sub(r'^nv(\d+)@([\d.]+)$', r'vol .01 equalizer \2 400h 55 compand .0001,.05 6:-\1.0001,-\1,-inf,0,-inf 0 0 .0015 '+sil+r' -1 .022 .001% equalizer \2 400h -55 vol 100', p)
			p = re.sub(r'^nv(\d+)$', r'compand .0001,.05 6:-\1.0001,-\1,-inf,0,-inf 0 0 .0015 '+sil, p)
			# ov, only-vocal, number is dB at and below which to cut out other noise.
			# This effect also trims out the resulting silence.
			# The first version allows for an arbitrary band to be preserved preferentially.
			# The second version allows for a 400h-wide band to be preserved preferentially.
			p = re.sub(r'^ov(\d+)@([\d.]+)/([\d.]+)$', r'vol .01 equalizer \2 \3h 55 compand .01,.1 -\1.01,-inf,-\1,-\1 0 0 .012 silence -l 1 .022 .001% -1 .7 .001% equalizer \2 \3h -55 vol 100', p)
			p = re.sub(r'^ov(\d+)@([\d.]+)$', r'vol .01 equalizer \2 400h 55 compand .01,.1 -\1.01,-inf,-\1,-\1 0 0 .012 silence -l 1 .022 .001% -1 .7 .001% equalizer \2 400h -55 vol 100', p)
			p = re.sub(r'^ov(\d+)$', r'compand .01,.1 -\1.01,-inf,-\1,-\1 0 0 .012 silence -l 1 .022 .01% -1 .7 .01%', p)
			# dtmf<string>
			p = re.sub(r'(?i)dtmf([0-9#*abcd/.]+)', self._dtmf, p)
			parts[i] = p
			# nr<start>,<duration>[,<factor>]: Noise reduction with on-the-fly profile.
			# Without factor or with a factor of 1[.0], noisered uses 0.0. If factor is less than 1, noisered uses that.
			# If factor is above 1, noisered uses 0.0 and a vol <factor> is prepended to the noisered effect.
			match = re.match(r'^nr([\d.:=]+),([\d.:=]+)(.*)$', p)
			if match:
				profstart = match.groups()[0]
				proflen = match.groups()[1]
				rest = match.groups()[2]
				fac = 0.0
				if rest: fac = float(rest[1:])
				vol = 1.0
				if fac >= 1.0:
					vol = fac
					fac = 0.0
				p = "noisered - {0}".format(fac)
				if vol != 1.0: p = "vol {0} {1}".format(vol, p)
				# Now for the noiseprof command.
				prof = "trim {0} {1} {2} noiseprof -".format(profstart, proflen, " ".join(parts[:i]))
				self.profEffects = prof
			parts[i] = p
		v = " ".join(parts)
		self.expanded = v

	def _dtmf(self, m):
		"""Return a SoX effect chain that would produce the string of DTMF digits given.
		Digits 1-9, 0, and A-D are supported.
		Each produced tone is about 200 ms long excluding tiny fade-in and fade-out times to prevent clicks.
		There is a 100 ms pause between digits, again excluding start and end fade times.
		The fade time at each end of each tone is 20 ms.
		There is also one second of silence after the end, produced by a separate effect chain separated from the main one by a colon.
		If this is not desirable, something like result.split(":", 1)[0].rstrip() will remove it.
		These four defaults can be changed by trailing specs separated by slashes in the given order; e.g.,
		/300/200/30/5, or //100 to only make tones closer together.
		"""
		rows = (697, 770, 852, 941)
		cols = (1209, 1336, 1477, 1633)
		order = "123a456b789c*0#d"
		parms = m.group(1).lower().split("/")
		seq = parms.pop(0)
		# This is the duration of the full-volume portion of a digit (excluding the fades at both ends).
		# This is long by modern standards but safer across fuzzy connections, such as the cell network.
		dur = self._dtmfDefault(parms, 0.2)
		# This is the inter-digit space. Due to fading at both ends of each digit, it may seem longer to the ear than is shown here.
		spacing = self._dtmfDefault(parms, 0.1)
		# This is the length of a hyperbolic fade applied at both start and end of each digit.
		# The actual length of a dialed digit is the above duration plus two times this fade length.
		fade = self._dtmfDefault(parms, 0.02)
		# This is the duration of a final silence after all digits.
		end = self._dtmfDefault(parms, 1)
		results = []
		for ch in seq:
			idx = order.index(ch)
			row,col = rows[idx//4], cols[idx%4]
			fullDur = str( float(dur) + 2*float(fade))
			# vol .9 below prevents clipping while maintaining a high output volume.
			results.append("synth {0} sine {1} sine {2} channels 1 vol .9 fade {3} {4} {3} : trim 0:0 0:{5}".format(
				fullDur, row, col,
				fade, dur, spacing
			))
		buf = " : ".join(results)
		# This prevents a ValueError caused by SoX exiting too fast on a single dialed digit.
		if buf: buf += f" : trim 0 {end}"
		return buf

	def _dtmfDefault(self, parms, dfl):
		val = dfl
		if not parms: return val
		parm = parms.pop(0)
		if parm == '': return val
		val = f"{float(parm) / 1000:f}"
		return val

